const { RestClientV5 } = require('bybit-api');
const config = require('./config');
const { logger, tradeLogger } = require('./utils/logger');
const telegram = require('./utils/telegram');
const riskManager = require('./utils/risk');
const gridStrategy = require('./strategies/gridTrading');
const rsiIndicator = require('./strategies/rsiIndicator');

class CryptoTradingBot {
  constructor() {
    this.client = null;
    this.isRunning = false;
    this.balance = config.trading.initialCapital;
    this.stats = {
      totalTrades: 0,
      wins: 0,
      losses: 0,
      totalProfit: 0,
      startTime: Date.now(),
      trades: []
    };
    this.lastSummaryTime = Date.now();
    this.priceCache = {}; // Cache de precios actuales
  }

  // Inicializar el bot
  async initialize() {
    try {
      logger.info('🚀 Iniciando bot de trading...');

      // Inicializar cliente de Bybit
      this.client = new RestClientV5({
        key: config.bybit.apiKey,
        secret: config.bybit.apiSecret,
        testnet: config.bybit.testnet
      });

      // Verificar conexión
      await this.verifyConnection();

      // Obtener balance inicial
      await this.updateBalance();

      // Inicializar grids para cada par
      for (const pair of config.trading.pairs) {
        const price = await this.getCurrentPrice(pair);
        await gridStrategy.initializeGrid(pair, price);
        logger.info(`✅ ${pair} inicializado a $${price.toFixed(2)}`);
      }

      // Enviar notificación de inicio
      await telegram.sendStartup(this.balance, config.trading.pairs);

      this.isRunning = true;
      logger.info('✅ Bot inicializado correctamente');
      
      return true;
    } catch (error) {
      logger.error(`Error inicializando bot: ${error.message}`);
      await telegram.sendError(error, 'Inicialización');
      throw error;
    }
  }

  // Verificar conexión con Bybit
  async verifyConnection() {
    try {
      const serverTime = await this.client.getServerTime();
      logger.info(`🌐 Conectado a Bybit ${config.bybit.testnet ? 'Testnet' : 'Mainnet'}`);
      logger.debug(`Tiempo del servidor: ${new Date(serverTime.result.timeSecond * 1000).toISOString()}`);
      return true;
    } catch (error) {
      throw new Error(`No se pudo conectar a Bybit: ${error.message}`);
    }
  }

  // Obtener balance actual
  async updateBalance() {
    try {
      const response = await this.client.getWalletBalance({
        accountType: 'UNIFIED'
      });

      if (response.retCode === 0) {
        const usdtBalance = response.result.list[0]?.coin?.find(c => c.coin === 'USDT');
        if (usdtBalance) {
          this.balance = parseFloat(usdtBalance.walletBalance);
          logger.debug(`💰 Balance actualizado: $${this.balance.toFixed(2)} USDT`);
        }
      }
    } catch (error) {
      logger.error(`Error obteniendo balance: ${error.message}`);
    }
  }

  // Obtener precio actual de un par
  async getCurrentPrice(symbol) {
    try {
      const response = await this.client.getTickers({
        category: config.trading.category,
        symbol: symbol
      });

      if (response.retCode === 0 && response.result.list.length > 0) {
        const price = parseFloat(response.result.list[0].lastPrice);
        this.priceCache[symbol] = price;
        
        // Agregar precio al historial de RSI
        rsiIndicator.addPrice(symbol, price);
        
        return price;
      }
      
      throw new Error('No se pudo obtener el precio');
    } catch (error) {
      logger.error(`Error obteniendo precio de ${symbol}: ${error.message}`);
      return this.priceCache[symbol] || 0;
    }
  }

  // Ejecutar una orden
  async executeOrder(symbol, side, qty, price) {
    if (config.system.dryRun) {
      logger.info(`[DRY RUN] Orden simulada: ${side} ${qty} ${symbol} @ $${price}`);
      return { success: true, orderId: 'DRY_RUN_' + Date.now() };
    }

    try {
      const orderParams = {
        category: config.trading.category,
        symbol: symbol,
        side: side,
        orderType: 'Market',
        qty: qty.toString(),
        timeInForce: 'GTC'
      };

      const response = await this.client.submitOrder(orderParams);

      if (response.retCode === 0) {
        logger.info(`✅ Orden ejecutada: ${side} ${qty} ${symbol} @ $${price}`);
        return { success: true, orderId: response.result.orderId };
      } else {
        throw new Error(response.retMsg);
      }
    } catch (error) {
      logger.error(`Error ejecutando orden: ${error.message}`);
      await telegram.sendError(error, `Ejecutar orden ${symbol}`);
      return { success: false, error: error.message };
    }
  }

  // Procesar señales de trading para un símbolo
  async processSymbol(symbol) {
    try {
      // Obtener precio actual
      const currentPrice = await this.getCurrentPrice(symbol);
      if (!currentPrice) return;

      // Verificar si se puede operar
      const riskCheck = riskManager.canOpenTrade(this.balance);
      if (!riskCheck.allowed) {
        logger.debug(`${symbol}: ${riskCheck.reason}`);
        return;
      }

      // Obtener señal del grid
      const gridSignal = gridStrategy.getSignal(symbol, currentPrice, this.balance);
      if (!gridSignal) return;

      // Verificar con RSI
      const rsiAllows = rsiIndicator.allowsTrade(symbol, gridSignal.side);
      if (!rsiAllows) {
        logger.debug(`${symbol}: RSI no permite ${gridSignal.side}`);
        return;
      }

      // Ejecutar la orden
      logger.info(`🎯 Señal detectada para ${symbol}: ${gridSignal.side} @ $${currentPrice.toFixed(2)}`);
      const result = await this.executeOrder(
        symbol,
        gridSignal.side,
        gridSignal.qty,
        currentPrice
      );

      if (result.success) {
        await this.recordTrade(symbol, gridSignal.side, currentPrice, gridSignal.qty);
      }

    } catch (error) {
      logger.error(`Error procesando ${symbol}: ${error.message}`);
    }
  }

  // Registrar un trade ejecutado
  async recordTrade(symbol, side, price, qty) {
    const trade = {
      symbol,
      side,
      price,
      qty,
      timestamp: Date.now(),
      value: price * qty
    };

    // Actualizar posición en grid
    if (side === 'Buy') {
      gridStrategy.openPosition(symbol, side, price, qty);
    } else {
      const position = gridStrategy.positions[symbol];
      if (position) {
        const profit = (price - position.price) * qty;
        trade.profit = profit;
        trade.profitPercent = (profit / (position.price * qty)) * 100;
        
        this.stats.totalProfit += profit;
        this.stats.totalTrades++;
        
        if (profit > 0) {
          this.stats.wins++;
        } else {
          this.stats.losses++;
        }

        riskManager.recordTrade(profit);
        gridStrategy.closePosition(symbol);
      }
    }

    // Guardar en log
    this.stats.trades.push(trade);
    tradeLogger.info('trade', trade);

    // Notificar
    await telegram.sendTrade(trade);

    logger.info(`📝 Trade registrado: ${side} ${qty} ${symbol} @ $${price}`);
  }

  // Enviar resumen periódico
  async sendPeriodicSummary() {
    const now = Date.now();
    const hoursSinceLastSummary = (now - this.lastSummaryTime) / (1000 * 60 * 60);

    if (hoursSinceLastSummary >= config.notifications.summaryIntervalHours) {
      const stats = this.getStats();
      await telegram.sendSummary(stats);
      this.lastSummaryTime = now;
    }
  }

  // Obtener estadísticas
  getStats() {
    const winRate = this.stats.totalTrades > 0 
      ? (this.stats.wins / this.stats.totalTrades) * 100 
      : 0;

    const profitPercent = (this.stats.totalProfit / config.trading.initialCapital) * 100;

    const profits = this.stats.trades
      .filter(t => t.profit)
      .map(t => t.profit);

    const bestTrade = profits.length > 0 ? Math.max(...profits) : 0;
    const worstTrade = profits.length > 0 ? Math.min(...profits) : 0;

    const nextSummary = new Date(
      this.lastSummaryTime + (config.notifications.summaryIntervalHours * 60 * 60 * 1000)
    );

    return {
      period: `${config.notifications.summaryIntervalHours}H`,
      balance: this.balance,
      totalProfit: this.stats.totalProfit,
      profitPercent,
      totalTrades: this.stats.totalTrades,
      wins: this.stats.wins,
      losses: this.stats.losses,
      winRate,
      bestTrade,
      worstTrade,
      nextSummary: nextSummary.toLocaleTimeString('es-MX')
    };
  }

  // Loop principal
  async run() {
    while (this.isRunning) {
      try {
        // Procesar cada par
        for (const symbol of config.trading.pairs) {
          await this.processSymbol(symbol);
          await this.sleep(1000); // 1 segundo entre pares
        }

        // Actualizar balance
        await this.updateBalance();

        // Enviar resumen si corresponde
        await this.sendPeriodicSummary();

        // Esperar antes del siguiente ciclo
        await this.sleep(config.system.checkIntervalSeconds * 1000);

      } catch (error) {
        logger.error(`Error en loop principal: ${error.message}`);
        await telegram.sendError(error, 'Loop principal');
        await this.sleep(30000); // Esperar 30 segundos antes de reintentar
      }
    }
  }

  // Detener el bot
  async stop() {
    this.isRunning = false;
    logger.info('🛑 Deteniendo bot...');
    
    const finalStats = this.getStats();
    await telegram.sendSummary({ ...finalStats, period: 'FINAL' });
    
    logger.info('✅ Bot detenido');
  }

  // Helper para dormir
  sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
}

// Manejo de señales para detener el bot
let bot = null;

async function startBot() {
  try {
    bot = new CryptoTradingBot();
    await bot.initialize();
    await bot.run();
  } catch (error) {
    logger.error(`Error fatal: ${error.message}`);
    process.exit(1);
  }
}

// Capturar CTRL+C
process.on('SIGINT', async () => {
  logger.info('Señal SIGINT recibida');
  if (bot) {
    await bot.stop();
  }
  process.exit(0);
});

// Capturar errores no manejados
process.on('unhandledRejection', (error) => {
  logger.error(`Error no manejado: ${error.message}`);
  logger.error(error.stack);
});

// Iniciar el bot
if (require.main === module) {
  startBot();
}

module.exports = CryptoTradingBot;
