#!/bin/bash

# Script de configuración inicial del bot de trading
# Ejecuta: bash setup.sh

echo "=========================================="
echo "🤖 BOT DE TRADING - CONFIGURACIÓN INICIAL"
echo "=========================================="
echo ""

# Verificar Node.js
echo "📦 Verificando Node.js..."
if ! command -v node &> /dev/null; then
    echo "❌ Node.js no está instalado"
    echo "   Instala Node.js 16+ desde: https://nodejs.org"
    exit 1
fi

NODE_VERSION=$(node -v | cut -d'v' -f2 | cut -d'.' -f1)
if [ "$NODE_VERSION" -lt 16 ]; then
    echo "⚠️  Node.js version $NODE_VERSION detectada"
    echo "   Se recomienda Node.js 16 o superior"
else
    echo "✅ Node.js $(node -v) detectado"
fi

# Verificar npm
if ! command -v npm &> /dev/null; then
    echo "❌ npm no está instalado"
    exit 1
fi
echo "✅ npm $(npm -v) detectado"
echo ""

# Instalar dependencias
echo "📦 Instalando dependencias..."
npm install

if [ $? -eq 0 ]; then
    echo "✅ Dependencias instaladas correctamente"
else
    echo "❌ Error instalando dependencias"
    exit 1
fi
echo ""

# Crear directorios necesarios
echo "📁 Creando directorios..."
mkdir -p logs data
echo "✅ Directorios creados"
echo ""

# Verificar archivo .env
echo "⚙️  Verificando configuración..."
if [ ! -f .env ]; then
    echo "❌ Archivo .env no encontrado"
    exit 1
fi

# Verificar si las API keys están configuradas
if grep -q "your_bybit_api_key_here" .env; then
    echo "⚠️  IMPORTANTE: Debes configurar tus API keys en .env"
    echo ""
    echo "1. Abre Bybit Testnet: https://testnet.bybit.com/app/user/api-management"
    echo "2. Crea una nueva API Key"
    echo "3. Copia la Key y Secret"
    echo "4. Edita el archivo .env y reemplaza:"
    echo "   BYBIT_API_KEY=your_bybit_api_key_here"
    echo "   BYBIT_API_SECRET=your_bybit_api_secret_here"
    echo ""
    echo "Después ejecuta de nuevo: bash setup.sh"
    exit 1
fi

echo "✅ API Keys configuradas"
echo ""

# Ofrecer ejecutar backtest
echo "🧪 ¿Deseas ejecutar un backtest ahora? (recomendado)"
echo "   Esto probará la estrategia con datos históricos"
read -p "   (s/n): " -n 1 -r
echo ""

if [[ $REPLY =~ ^[Ss]$ ]]; then
    echo "📊 Ejecutando backtest..."
    npm run backtest
    echo ""
fi

echo "=========================================="
echo "✅ CONFIGURACIÓN COMPLETADA"
echo "=========================================="
echo ""
echo "Para iniciar el bot, ejecuta uno de estos comandos:"
echo ""
echo "  1. Modo simple (para pruebas):"
echo "     npm start"
echo ""
echo "  2. Con screen (recomendado para VPS):"
echo "     screen -S crypto-bot"
echo "     npm start"
echo "     # Presiona Ctrl+A, luego D para desconectar"
echo ""
echo "  3. Con PM2 (más profesional):"
echo "     npm install -g pm2"
echo "     pm2 start bot.js --name crypto-bot"
echo "     pm2 logs crypto-bot"
echo ""
echo "📱 Recibirás notificaciones en Telegram"
echo "📊 Logs disponibles en: logs/bot.log"
echo ""
echo "¡Buena suerte con el trading! 🚀"
