const config = require('../config');
const { logger } = require('../utils/logger');
const riskManager = require('../utils/risk');

class GridTradingStrategy {
  constructor() {
    this.grids = {}; // { symbol: { levels, currentLevel, orders } }
    this.positions = {}; // { symbol: { price, qty, side } }
  }

  // Inicializar grid para un símbolo
  async initializeGrid(symbol, currentPrice) {
    const rangePercent = config.grid.rangePercent / 100;
    const numLevels = config.grid.levels;

    // Calcular rango de precios
    const lowerBound = currentPrice * (1 - rangePercent);
    const upperBound = currentPrice * (1 + rangePercent);
    const step = (upperBound - lowerBound) / (numLevels - 1);

    // Crear niveles del grid
    const levels = [];
    for (let i = 0; i < numLevels; i++) {
      const price = lowerBound + (step * i);
      levels.push({
        price: parseFloat(price.toFixed(2)),
        type: i < numLevels / 2 ? 'buy' : 'sell',
        executed: false
      });
    }

    this.grids[symbol] = {
      levels,
      currentLevel: this.findCurrentLevel(levels, currentPrice),
      lowerBound,
      upperBound,
      lastUpdate: Date.now()
    };

    logger.info(`📊 Grid inicializado para ${symbol}: ${numLevels} niveles entre $${lowerBound.toFixed(2)} - $${upperBound.toFixed(2)}`);
    return this.grids[symbol];
  }

  // Encontrar nivel actual del grid
  findCurrentLevel(levels, price) {
    let closest = 0;
    let minDiff = Math.abs(levels[0].price - price);

    for (let i = 1; i < levels.length; i++) {
      const diff = Math.abs(levels[i].price - price);
      if (diff < minDiff) {
        minDiff = diff;
        closest = i;
      }
    }

    return closest;
  }

  // Verificar si el precio alcanzó un nuevo nivel
  checkLevelCrossed(symbol, currentPrice) {
    const grid = this.grids[symbol];
    if (!grid) return null;

    const newLevel = this.findCurrentLevel(grid.levels, currentPrice);

    // Si cambió de nivel
    if (newLevel !== grid.currentLevel) {
      const level = grid.levels[newLevel];
      const previousLevel = grid.currentLevel;
      
      grid.currentLevel = newLevel;

      logger.info(`🎯 ${symbol} cruzó nivel ${newLevel}: $${level.price} (${level.type})`);

      return {
        level: newLevel,
        price: level.price,
        type: level.type,
        previousLevel,
        direction: newLevel > previousLevel ? 'up' : 'down'
      };
    }

    return null;
  }

  // Obtener señal de trading basada en grid
  getSignal(symbol, currentPrice, balance) {
    const grid = this.grids[symbol];
    if (!grid) return null;

    const levelCrossed = this.checkLevelCrossed(symbol, currentPrice);
    if (!levelCrossed) return null;

    const { type, direction, level } = levelCrossed;

    // Señal de compra: precio bajó y llegó a nivel de compra
    if (type === 'buy' && direction === 'down') {
      const positionSize = riskManager.calculatePositionSize(balance, currentPrice);
      
      return {
        side: 'Buy',
        price: currentPrice,
        qty: positionSize,
        level,
        reason: `Grid nivel ${level} alcanzado (compra)`
      };
    }

    // Señal de venta: precio subió y llegó a nivel de venta
    if (type === 'sell' && direction === 'up') {
      // Solo vender si tenemos posición
      const position = this.positions[symbol];
      if (!position || position.side !== 'Buy') {
        return null;
      }

      return {
        side: 'Sell',
        price: currentPrice,
        qty: position.qty,
        level,
        reason: `Grid nivel ${level} alcanzado (venta)`
      };
    }

    return null;
  }

  // Registrar posición abierta
  openPosition(symbol, side, price, qty) {
    this.positions[symbol] = {
      side,
      price: parseFloat(price),
      qty: parseFloat(qty),
      openTime: Date.now()
    };

    logger.debug(`Posición abierta: ${symbol} ${side} ${qty} @ $${price}`);
  }

  // Cerrar posición
  closePosition(symbol) {
    if (this.positions[symbol]) {
      delete this.positions[symbol];
      logger.debug(`Posición cerrada: ${symbol}`);
    }
  }

  // Verificar si necesita ajustar el grid por volatilidad
  shouldAdjustGrid(symbol, volatility) {
    const grid = this.grids[symbol];
    if (!grid) return false;

    // Si la volatilidad es muy alta (>5%), ajustar grid
    if (volatility > 5) {
      return { adjust: true, reason: 'volatilidad alta', newRange: config.grid.rangePercent * 1.5 };
    }

    // Si la volatilidad es muy baja (<0.5%), ajustar grid
    if (volatility < 0.5) {
      return { adjust: true, reason: 'volatilidad baja', newRange: config.grid.rangePercent * 0.7 };
    }

    return { adjust: false };
  }

  // Ajustar grid basado en volatilidad
  async adjustGrid(symbol, currentPrice, volatility) {
    const adjustment = this.shouldAdjustGrid(symbol, volatility);
    
    if (adjustment.adjust) {
      logger.info(`🔄 Ajustando grid de ${symbol} por ${adjustment.reason}`);
      
      // Guardar configuración temporal
      const originalRange = config.grid.rangePercent;
      config.grid.rangePercent = adjustment.newRange;
      
      // Reinicializar grid
      await this.initializeGrid(symbol, currentPrice);
      
      // Restaurar configuración
      config.grid.rangePercent = originalRange;
      
      return true;
    }

    return false;
  }

  // Obtener información del grid
  getGridInfo(symbol) {
    const grid = this.grids[symbol];
    if (!grid) return null;

    const currentLevel = grid.levels[grid.currentLevel];
    const position = this.positions[symbol];

    return {
      symbol,
      currentLevel: grid.currentLevel,
      currentPrice: currentLevel.price,
      totalLevels: grid.levels.length,
      range: `$${grid.lowerBound.toFixed(2)} - $${grid.upperBound.toFixed(2)}`,
      hasPosition: !!position,
      position: position || null
    };
  }

  // Obtener todas las posiciones activas
  getActivePositions() {
    return Object.entries(this.positions).map(([symbol, pos]) => ({
      symbol,
      ...pos
    }));
  }

  // Calcular ganancia/pérdida de una posición
  calculatePnL(symbol, currentPrice) {
    const position = this.positions[symbol];
    if (!position) return 0;

    const entryValue = position.price * position.qty;
    const currentValue = currentPrice * position.qty;

    if (position.side === 'Buy') {
      return currentValue - entryValue;
    } else {
      return entryValue - currentValue;
    }
  }

  // Limpiar grid de un símbolo
  clearGrid(symbol) {
    if (this.grids[symbol]) {
      delete this.grids[symbol];
      logger.debug(`Grid de ${symbol} limpiado`);
    }
  }
}

module.exports = new GridTradingStrategy();
